/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
#include "first.h"

static CURLcode test_lib525(const char *URL)
{
  CURLcode result = CURLE_OK;
  CURL *curl = NULL;
  char errbuf[STRERROR_LEN];
  FILE *hd_src = NULL;
  int hd;
  curl_struct_stat file_info;
  CURLM *multi = NULL;
  int running;

  start_test_timing();

  if(!libtest_arg2) {
    curl_mfprintf(stderr, "Usage: test [url] [uploadfile]\n");
    return TEST_ERR_USAGE;
  }

  hd_src = curlx_fopen(libtest_arg2, "rb");
  if(!hd_src) {
    curl_mfprintf(stderr, "fopen() failed with error (%d) %s\n",
                  errno, curlx_strerror(errno, errbuf, sizeof(errbuf)));
    curl_mfprintf(stderr, "Error opening file '%s'\n", libtest_arg2);
    return TEST_ERR_FOPEN;
  }

  /* get the file size of the local file */
  hd = curlx_fstat(fileno(hd_src), &file_info);
  if(hd == -1) {
    /* cannot open file, bail out */
    curl_mfprintf(stderr, "fstat() failed with error (%d) %s\n",
                  errno, curlx_strerror(errno, errbuf, sizeof(errbuf)));
    curl_mfprintf(stderr, "Error opening file '%s'\n", libtest_arg2);
    curlx_fclose(hd_src);
    return TEST_ERR_FSTAT;
  }

  res_global_init(CURL_GLOBAL_ALL);
  if(result) {
    curlx_fclose(hd_src);
    return result;
  }

  easy_init(curl);

  /* enable uploading */
  easy_setopt(curl, CURLOPT_UPLOAD, 1L);

  /* specify target */
  easy_setopt(curl, CURLOPT_URL, URL);

  /* go verbose */
  easy_setopt(curl, CURLOPT_VERBOSE, 1L);

  /* use active FTP */
  easy_setopt(curl, CURLOPT_FTPPORT, "-");

  /* now specify which file to upload */
  easy_setopt(curl, CURLOPT_READDATA, hd_src);

  /* NOTE: if you want this code to work on Windows with libcurl as a DLL, you
     MUST also provide a read callback with CURLOPT_READFUNCTION. Failing to
     do so will give you a crash since a DLL may not use the variable's memory
     when passed in to it from an app like this. */

  /* Set the size of the file to upload (optional).  If you give a *_LARGE
     option you MUST make sure that the type of the passed-in argument is a
     curl_off_t. If you use CURLOPT_INFILESIZE (without _LARGE) you must
     make sure that to pass in a type 'long' argument. */
  easy_setopt(curl, CURLOPT_INFILESIZE_LARGE, (curl_off_t)file_info.st_size);

  multi_init(multi);

  multi_add_handle(multi, curl);

  for(;;) {
    struct timeval interval;
    fd_set rd, wr, exc;
    int maxfd = -99;

    interval.tv_sec = 1;
    interval.tv_usec = 0;

    multi_perform(multi, &running);

    abort_on_test_timeout();

    if(!running)
      break; /* done */

    FD_ZERO(&rd);
    FD_ZERO(&wr);
    FD_ZERO(&exc);

    multi_fdset(multi, &rd, &wr, &exc, &maxfd);

    /* At this point, maxfd is guaranteed to be greater or equal than -1. */

    select_test(maxfd + 1, &rd, &wr, &exc, &interval);

    abort_on_test_timeout();
  }

test_cleanup:

  if(testnum == 529) {
    /* proper cleanup sequence - type PA */
    curl_multi_remove_handle(multi, curl);
    curl_multi_cleanup(multi);
    curl_easy_cleanup(curl);
    curl_global_cleanup();
  }
  else { /* testnum == 525 */
    /* proper cleanup sequence - type PB */
    curl_multi_remove_handle(multi, curl);
    curl_easy_cleanup(curl);
    curl_multi_cleanup(multi);
    curl_global_cleanup();
  }

  /* close the local file */
  curlx_fclose(hd_src);

  return result;
}
