/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
#include "first.h"

#define MAX_EASY_HANDLES 3

static int ntlm_counter[MAX_EASY_HANDLES];
static CURL *ntlm_curls[MAX_EASY_HANDLES];
static curl_socket_t ntlm_sockets[MAX_EASY_HANDLES];
static CURLcode ntlmcb_res = CURLE_OK;

static size_t callback(char *ptr, size_t size, size_t nmemb, void *data)
{
  ssize_t idx = ((CURL **)data) - ntlm_curls;
  curl_socket_t sock;
  long longdata;
  CURLcode code;
  const size_t failure = (size && nmemb) ? 0 : 1;
  (void)ptr;

  ntlm_counter[idx] += (int)(size * nmemb);

  /* Get socket being used for this easy handle, otherwise CURL_SOCKET_BAD */
  code = curl_easy_getinfo(ntlm_curls[idx], CURLINFO_LASTSOCKET, &longdata);

  if(CURLE_OK != code) {
    curl_mfprintf(stderr, "%s:%d curl_easy_getinfo() failed, "
                  "with code %d (%s)\n",
                  __FILE__, __LINE__, code, curl_easy_strerror(code));
    ntlmcb_res = TEST_ERR_MAJOR_BAD;
    return failure;
  }
  if(longdata == -1L)
    sock = CURL_SOCKET_BAD;
  else
    sock = (curl_socket_t)longdata;

  if(sock != CURL_SOCKET_BAD) {
    /* Track relationship between this easy handle and the socket. */
    if(ntlm_sockets[idx] == CURL_SOCKET_BAD) {
      /* An easy handle without previous socket, record the socket. */
      ntlm_sockets[idx] = sock;
    }
    else if(sock != ntlm_sockets[idx]) {
      /* An easy handle with a socket different to previously
         tracked one, log and fail right away. Known bug #37. */
      curl_mfprintf(stderr, "Handle %zd started on socket %" FMT_SOCKET_T
                    " and moved to %" FMT_SOCKET_T "\n",
                    idx, ntlm_sockets[idx], sock);
      ntlmcb_res = TEST_ERR_MAJOR_BAD;
      return failure;
    }
  }
  return size * nmemb;
}

static CURLcode test_lib2032(const char *URL)  /* libntlmconnect */
{
  enum HandleState {
    ReadyForNewHandle,
    NeedSocketForNewHandle,
    NoMoreHandles
  };

  CURLcode res = CURLE_OK;
  CURLM *multi = NULL;
  int running;
  int i;
  int num_handles = 0;
  enum HandleState state = ReadyForNewHandle;
  size_t urllen = strlen(URL) + 4 + 1;
  char *full_url = curlx_malloc(urllen);

  start_test_timing();

  if(!full_url) {
    curl_mfprintf(stderr, "Not enough memory for full url\n");
    return TEST_ERR_MAJOR_BAD;
  }

  for(i = 0; i < MAX_EASY_HANDLES; ++i) {
    ntlm_curls[i] = NULL;
    ntlm_sockets[i] = CURL_SOCKET_BAD;
  }

  res_global_init(CURL_GLOBAL_ALL);
  if(res) {
    curlx_free(full_url);
    return res;
  }

  multi_init(multi);

  for(;;) {
    struct timeval interval;
    fd_set fdread;
    fd_set fdwrite;
    fd_set fdexcep;
    long timeout = -99;
    int maxfd = -99;
    bool found_new_socket = FALSE;

    /* Start a new handle if we are not at the max */
    if(state == ReadyForNewHandle) {
      easy_init(ntlm_curls[num_handles]);

      if(num_handles % 3 == 2) {
        curl_msnprintf(full_url, urllen, "%s0200", URL);
        easy_setopt(ntlm_curls[num_handles], CURLOPT_HTTPAUTH, CURLAUTH_NTLM);
      }
      else {
        curl_msnprintf(full_url, urllen, "%s0100", URL);
        easy_setopt(ntlm_curls[num_handles], CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
      }
      easy_setopt(ntlm_curls[num_handles], CURLOPT_FRESH_CONNECT, 1L);
      easy_setopt(ntlm_curls[num_handles], CURLOPT_URL, full_url);
      easy_setopt(ntlm_curls[num_handles], CURLOPT_VERBOSE, 1L);
      easy_setopt(ntlm_curls[num_handles], CURLOPT_HTTPGET, 1L);
      easy_setopt(ntlm_curls[num_handles], CURLOPT_USERPWD,
                  "testuser:testpass");
      easy_setopt(ntlm_curls[num_handles], CURLOPT_WRITEFUNCTION, callback);
      easy_setopt(ntlm_curls[num_handles], CURLOPT_WRITEDATA,
                  (void *)(ntlm_curls + num_handles));
      easy_setopt(ntlm_curls[num_handles], CURLOPT_HEADER, 1L);

      multi_add_handle(multi, ntlm_curls[num_handles]);
      num_handles += 1;
      state = NeedSocketForNewHandle;
      res = ntlmcb_res;
    }

    multi_perform(multi, &running);

    curl_mfprintf(stderr, "%s:%d running %d state %d\n",
                  __FILE__, __LINE__, running, state);

    abort_on_test_timeout();

    if(!running && state == NoMoreHandles)
      break; /* done */

    FD_ZERO(&fdread);
    FD_ZERO(&fdwrite);
    FD_ZERO(&fdexcep);

    multi_fdset(multi, &fdread, &fdwrite, &fdexcep, &maxfd);

    /* At this point, maxfd is guaranteed to be greater or equal than -1. */

    if(state == NeedSocketForNewHandle) {
      if(maxfd != -1 && !found_new_socket) {
        curl_mfprintf(stderr,
                      "Warning: socket did not open immediately for new "
                      "handle (trying again)\n");
        continue;
      }
      state = num_handles < MAX_EASY_HANDLES ? ReadyForNewHandle
                                             : NoMoreHandles;
      curl_mfprintf(stderr, "%s:%d new state %d\n",
                    __FILE__, __LINE__, state);
    }

    multi_timeout(multi, &timeout);

    /* At this point, timeout is guaranteed to be greater or equal than -1. */

    curl_mfprintf(stderr, "%s:%d num_handles %d timeout %ld running %d\n",
                  __FILE__, __LINE__, num_handles, timeout, running);

    if(timeout != -1L) {
      int itimeout;
#if LONG_MAX > INT_MAX
      itimeout = (timeout > (long)INT_MAX) ? INT_MAX : (int)timeout;
#else
      itimeout = (int)timeout;
#endif
      interval.tv_sec = itimeout / 1000;
      interval.tv_usec = (itimeout % 1000) * 1000;
    }
    else {
      interval.tv_sec = 0;
      interval.tv_usec = 5000;

      /* if there is no timeout and we get here on the last handle, we may
         already have read the last part of the stream so waiting makes no
         sense */
      if(!running && num_handles == MAX_EASY_HANDLES) {
        break;
      }
    }

    select_test(maxfd + 1, &fdread, &fdwrite, &fdexcep, &interval);

    abort_on_test_timeout();
  }

test_cleanup:

  /* proper cleanup sequence - type PB */

  for(i = 0; i < MAX_EASY_HANDLES; i++) {
    curl_mprintf("Data connection %d: %d\n", i, ntlm_counter[i]);
    curl_multi_remove_handle(multi, ntlm_curls[i]);
    curl_easy_cleanup(ntlm_curls[i]);
  }

  curl_multi_cleanup(multi);
  curl_global_cleanup();

  curlx_free(full_url);

  return res;
}
