/***************************************************************************
 *                                  _   _ ____  _
 *  Project                     ___| | | |  _ \| |
 *                             / __| | | | |_) | |
 *                            | (__| |_| |  _ <| |___
 *                             \___|\___/|_| \_\_____|
 *
 * Copyright (C) Daniel Stenberg, <daniel@haxx.se>, et al.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution. The terms
 * are also available at https://curl.se/docs/copyright.html.
 *
 * You may opt to use, copy, modify, merge, publish, distribute and/or sell
 * copies of the Software, and permit persons to whom the Software is
 * furnished to do so, under the terms of the COPYING file.
 *
 * This software is distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY
 * KIND, either express or implied.
 *
 * SPDX-License-Identifier: curl
 *
 ***************************************************************************/
/* <DESC>
 * Multiplexed HTTP/2 downloads over a single connection
 * </DESC>
 */
#ifdef _MSC_VER
#ifndef _CRT_SECURE_NO_WARNINGS
#define _CRT_SECURE_NO_WARNINGS  /* for _snprintf(), fopen(), strerror() */
#endif
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#include <curl/curl.h>

#ifndef CURLPIPE_MULTIPLEX
/* This little trick makes sure that we do not enable pipelining for libcurls
   old enough to not have this symbol. It is _not_ defined to zero in a recent
   libcurl header. */
#define CURLPIPE_MULTIPLEX 0L
#endif

#if defined(_MSC_VER) && (_MSC_VER < 1900)
#define snprintf _snprintf
#endif

struct transfer {
  FILE *out;
  CURL *curl;
  int num;
};

static void dump(const char *text, int num, unsigned char *ptr,
                 size_t size, char nohex)
{
  size_t i;
  size_t c;
  unsigned int width = 0x10;

  if(nohex)
    /* without the hex output, we can fit more on screen */
    width = 0x40;

  fprintf(stderr, "%d %s, %lu bytes (0x%lx)\n",
          num, text, (unsigned long)size, (unsigned long)size);

  for(i = 0; i < size; i += width) {

    fprintf(stderr, "%4.4lx: ", (unsigned long)i);

    if(!nohex) {
      /* hex not disabled, show it */
      for(c = 0; c < width; c++)
        if(i + c < size)
          fprintf(stderr, "%02x ", ptr[i + c]);
        else
          fputs("   ", stderr);
    }

    for(c = 0; (c < width) && (i + c < size); c++) {
      /* check for 0D0A; if found, skip past and start a new line of output */
      if(nohex && (i + c + 1 < size) && ptr[i + c] == 0x0D &&
         ptr[i + c + 1] == 0x0A) {
        i += (c + 2 - width);
        break;
      }
      fprintf(stderr, "%c",
              (ptr[i + c] >= 0x20) && (ptr[i + c] < 0x80) ? ptr[i + c] : '.');
      /* check again for 0D0A, to avoid an extra \n if it is at width */
      if(nohex && (i + c + 2 < size) && ptr[i + c + 1] == 0x0D &&
         ptr[i + c + 2] == 0x0A) {
        i += (c + 3 - width);
        break;
      }
    }
    fputc('\n', stderr); /* newline */
  }
}

static int my_trace(CURL *curl, curl_infotype type,
                    char *data, size_t size, void *userp)
{
  const char *text;
  struct transfer *t = (struct transfer *)userp;
  int num = t->num;
  (void)curl;

  switch(type) {
  case CURLINFO_TEXT:
    fprintf(stderr, "== [%d] Info: %s", num, data);
    return 0;
  case CURLINFO_HEADER_OUT:
    text = "=> Send header";
    break;
  case CURLINFO_DATA_OUT:
    text = "=> Send data";
    break;
  case CURLINFO_SSL_DATA_OUT:
    text = "=> Send SSL data";
    break;
  case CURLINFO_HEADER_IN:
    text = "<= Recv header";
    break;
  case CURLINFO_DATA_IN:
    text = "<= Recv data";
    break;
  case CURLINFO_SSL_DATA_IN:
    text = "<= Recv SSL data";
    break;
  default: /* in case a new one is introduced to shock us */
    return 0;
  }

  dump(text, num, (unsigned char *)data, size, 1);
  return 0;
}

static int setup(struct transfer *t, int num)
{
  char filename[128];
  CURL *curl;

  curl = t->curl = NULL;

  t->num = num;
  snprintf(filename, sizeof(filename), "dl-%d", num);
  t->out = fopen(filename, "wb");
  if(!t->out) {
    fprintf(stderr, "error: could not open file %s for writing: %s\n",
            filename, strerror(errno));
    return 1;
  }

  curl = t->curl = curl_easy_init();
  if(curl) {

    /* write to this file */
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, t->out);

    /* set the same URL */
    curl_easy_setopt(curl, CURLOPT_URL, "https://localhost:8443/index.html");

    /* please be verbose */
    curl_easy_setopt(curl, CURLOPT_VERBOSE, 1L);
    curl_easy_setopt(curl, CURLOPT_DEBUGFUNCTION, my_trace);
    curl_easy_setopt(curl, CURLOPT_DEBUGDATA, t);

    /* enlarge the receive buffer for potentially higher transfer speeds */
    curl_easy_setopt(curl, CURLOPT_BUFFERSIZE, 100000L);

    /* HTTP/2 please */
    curl_easy_setopt(curl, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_2_0);

#if (CURLPIPE_MULTIPLEX > 0)
    /* wait for pipe connection to confirm */
    curl_easy_setopt(curl, CURLOPT_PIPEWAIT, 1L);
#endif
  }
  return 0;
}

/*
 * Download many transfers over HTTP/2, using the same connection!
 */
int main(int argc, char **argv)
{
  CURLcode res;
  struct transfer *trans;
  CURLM *multi = NULL;
  int i;
  int still_running = 0; /* keep number of running handles */
  int num_transfers;

  if(argc > 1) {
    /* if given a number, do that many transfers */
    num_transfers = atoi(argv[1]);
    if((num_transfers < 1) || (num_transfers > 1000))
      num_transfers = 3;  /* a suitable low default */
  }
  else
    num_transfers = 3;  /* a suitable low default */

  res = curl_global_init(CURL_GLOBAL_ALL);
  if(res)
    return (int)res;

  trans = calloc(num_transfers, sizeof(*trans));
  if(!trans) {
    fprintf(stderr, "error allocating transfer structs\n");
    goto error;
  }

  /* init a multi stack */
  multi = curl_multi_init();
  if(!multi)
    goto error;

  for(i = 0; i < num_transfers; i++) {
    if(setup(&trans[i], i)) {
      goto error;
    }

    /* add the individual transfer */
    curl_multi_add_handle(multi, trans[i].curl);
  }

  curl_multi_setopt(multi, CURLMOPT_PIPELINING, CURLPIPE_MULTIPLEX);

  do {
    CURLMcode mc = curl_multi_perform(multi, &still_running);

    if(still_running)
      /* wait for activity, timeout or "nothing" */
      mc = curl_multi_poll(multi, NULL, 0, 1000, NULL);

    if(mc)
      break;

  } while(still_running);

error:

  if(multi) {
    for(i = 0; i < num_transfers; i++) {
      curl_multi_remove_handle(multi, trans[i].curl);
      curl_easy_cleanup(trans[i].curl);

      if(trans[i].out)
        fclose(trans[i].out);
    }
    curl_multi_cleanup(multi);
  }

  free(trans);

  curl_global_cleanup();

  return 0;
}
